<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\CustomerWholesalePlan;
use App\Services\CashBackService;

class ProcessMonthlyCashback extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cashback:process-month';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process cashback for each active plan using stored cashback window and roll it forward';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle(CashBackService $cashBackService)
    {
        // prevent overlap if cron fires twice
        $lock = Cache::lock('cashback:process-month', 600); // 10 minutes
        if (!$lock->get()) {
            $this->info('Another cashback run is active. Exiting.');
            return self::SUCCESS;
        }
        try {
            $today = Carbon::now();
            $plansQuery = CustomerWholesalePlan::with('customer')
                ->where('status', ACTIVE)
                ->whereNotNull('cashback_from_date')
                ->whereNotNull('cashback_to_date')
                ->whereDate('cashback_to_date', '<=', $today->copy()->subDay()->toDateString());
            $plansQuery->chunkById(200, function ($plans) use ($cashBackService, $today) {
                foreach ($plans as $plan) {
                    DB::transaction(function () use ($cashBackService, $plan, $today) {
                        $customer = $plan->customer;
                        if (!$customer) {
                            return;
                        }
                        $from = Carbon::parse($plan->cashback_from_date);
                        $to = Carbon::parse($plan->cashback_to_date)->endOfDay();
                        $cashBackService->processCashBack($plan, $customer, $from, $to);
                        $plan->update([
                            'cashback_from_date' => $today->startOfMonth(),
                            'cashback_to_date' => $today->endOfMonth()->endOfDay(),
                        ]);
                    });
                }
            });
            $this->info('Cashback processed and windows rolled forward.');
        } finally {
            optional($lock)->release();
        }
        return self::SUCCESS;
    }
}
