<?php

namespace App\Services;

use App\Models\Customer;
use App\Models\CustomerWholesalePlan;
use App\Models\order;
use App\Models\Product;
use App\Models\Setting;
use App\Models\WholesaleIdPlan;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CashBackService
{
    public function assignBasicPlan($customer)
    {
        //if active plan then do nothing
        if ($customer->activePlanWholesale) {
            return $customer->activePlanWholesale;
        }
        //fetch basic plan
        $basicPlan = WholesaleIdPlan::where('is_default', 1)->first();
        if ($basicPlan) {
            $customPlanId = $this->getOrCreateCustomPlanIdForCustomer($customer->id);
            $startDate = Carbon::now();
            $endDate = $startDate->copy()->addYear();
            $plan =  CustomerWholesalePlan::create([
                'customer_id' => $customer->id,
                'original_plan_id' => $basicPlan->id,
                'custom_plan_id' => $customPlanId,
                'title' => $basicPlan->title,
                'price_in_usd' => $basicPlan->price_in_usd,
                'price_in_mmk' => $basicPlan->price_in_mmk,
                'price_in_rmb' => $basicPlan->price_in_rmb,
                'min_order_amt_usd' => $basicPlan->min_order_amt_usd,
                'min_order_amt_mmk' => $basicPlan->min_order_amt_mmk,
                'min_order_amt_rmb' => $basicPlan->min_order_amt_rmb,
                'discount' => $basicPlan->discount,
                'number_of_days' => 0,
                'start_date' => $startDate->toDateTimeString(),
                'end_date' => $endDate->toDateTimeString(),
                'status' => ACTIVE,
                'cashback_from_date' => $startDate,
                'cashback_to_date'   => $startDate->copy()->endOfMonth(),
            ]);
            Customer::where('id', $customer->id)->update(['wholesale_plan_id' => $customPlanId]);
            $customer->refresh();
            return $plan;
        }
        return null;
    }
    //check that customer have any active plan before on assign first time to him
    public function getOrCreateCustomPlanIdForCustomer($customerId)
    {
        $existing = CustomerWholesalePlan::where('customer_id', $customerId)->value('custom_plan_id');
        if ($existing) {
            return $existing;
        }
        return 'WCP-' . now()->format('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
    }
    //check existing active or scheduled plan for customer
    public function  deactivateExistingPlans($customerId)
    {
        $extistingUsePlans = CustomerWholesalePlan::where('customer_id', $customerId)->whereIn('status', [ACTIVE, SCHEDULED])->get();
        //set canceled or deactivate plans
        foreach ($extistingUsePlans as $extistingUsePlan) {
            if ($extistingUsePlan->status == ACTIVE) {
                $extistingUsePlan->status = DEACTIVE;
            } elseif ($extistingUsePlan->status == SCHEDULED) {
                $extistingUsePlan->status = CANCELLED;
            }
            $extistingUsePlan->save();
        }
    }
    //handle the cashback process whenever need it(first of month or when new plan buy)
    public static function processCashBack($plan, $customer, $fromDate, $ToDate)
    {
        $currenyInfo =  self::currencyWiseInfo($customer->currency);
        //get all orderswhich fall under requested date
        $orders = order::where('customer_id', $customer->id)
            ->whereBetween('created_at', [$fromDate, $ToDate])
            ->get();

        $customerCurrency = $customer->currency;
        //convert all orders's amount which fall under requested date in customer's currency
        $totalValue = $orders->sum(function ($order) use ($customerCurrency) {
            $orderAmountInUsd = $order->total_price / $order->conversion_rate;
            return self::convertUsdtoCustomerCurrency($orderAmountInUsd, $customerCurrency);
        });
        //get existing plan's min order amount
        $existingPlanMinOrderAmount = $plan->$currenyInfo['minOrderAmountField'];
        //check if reach amount then credit to wallet
        if ($totalValue >= $existingPlanMinOrderAmount) {
            //give cashback
            $cashbackAmount = ($totalValue * $plan->discount) / 100;
            //call cashbackwallet function
            //$this->creditToWallet($customer,$cashbackAmount)
            Log::info("Cashback Calcuated:", [
                'customer_id' => $customer->id,
                'plan_id'     => $plan->id,
                'from_date'   => $fromDate,
                'to_date'     => $ToDate,
                'order_total' => $totalValue,
                'cashback'    => $cashbackAmount,
            ]);
            return $cashbackAmount;
        }
        //not eligible
        return 0;
    }
    public static function convertUsdtoCustomerCurrency($orderAmountInUsd, $customerCurrency)
    {
        $currrncyRate = Setting::getValue(['usd_to_mmk', 'usd_to_rmb']);
        $currrncyRateMmk = $currrncyRate['usd_to_mmk'] ?? null;
        $currrncyRateRmb = $currrncyRate['usd_to_rmb'] ?? null;
        switch (strtolower($customerCurrency)) {
            case 'usd':
                return $orderAmountInUsd;
            case 'mmk':
                return $orderAmountInUsd * $currrncyRateMmk;
            case 'rmb':
                return $orderAmountInUsd * $currrncyRateRmb;
        }
    }
    //fetch cashback list with price and min order value according to customer currency
    public static function currencyWiseInfo($customerCurrency)
    {
        $currency = strtolower($customerCurrency);
        return [
            'priceField' => "price_in_{$currency}",
            'minOrderAmountField' => "min_order_amt_{$currency}",
        ];
    }
}
