<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Throwable;
use App\Helpers\CommonHelper;
use App\Helpers\NotificationHelper;
use App\Models\Customer;
use App\Models\Notification;
use App\Models\order;
use App\Models\orderDetail;
use App\Models\ProductReview;
use App\Models\User;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class ReviewRatingController extends Controller
{
    public function store(Request $request)
    {
        DB::beginTransaction();
        $customerId = auth()->user()->id;
        $allowedExtensions = array_merge(
            CommonHelper::allowedImageExtension(),
            CommonHelper::allowedVideoExtension()
        );
        $allowedMimes = implode(',', $allowedExtensions);
        try {
            $validation = Validator::make(
                $request->all(),
                [
                    'product_id' => 'required|exists:products,id',
                    'order_id' => 'required|exists:orders,id',
                    'rating' => 'required|numeric|min:0|max:5',
                    'review' => 'nullable|string',
                    'attachments' => 'nullable|array|max:5',
                    'attachments.*' => 'nullable|file|mimes:' . $allowedMimes . 'max:20480',
                    'delete_attachment_ids' => 'nullable|array',
                    'delete_attachment_ids.*' => 'exists:review_attachements,id'
                ],
                [
                    'attachments.*.max' => __('each_file_size_vald'),
                    'attachments.*.mimes' => __('file_type_vald'),
                ]
            );
            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors()); // Bad Request
            }
            $productId = $request->product_id;
            $orderId = $request->order_id;
            //check that puchase is verified only then proceed further
            $order = order::where('id', $orderId)
                ->where('customer_id', $customerId)
                ->where('order_status', 3)
                ->whereHas('orderDetails', function ($q) use ($productId) {
                    $q->where('product_id', $productId);
                })->first();
            if (!$order) {
                return CommonHelper::apiResponse(400, __('not_verified_purchased'));
            }
            //fecth review for delete attachement
            $existingReview = ProductReview::where('order_id', $orderId)
                ->where('product_id', $productId)
                ->where('customer_id', $customerId)
                ->first();
            //delete attachement if any
            if ($existingReview && $request->delete_attachment_ids) {
                $attachements =  $existingReview->attachments()
                    ->whereIn('id', $request->delete_attachment_ids)
                    ->get();
                foreach ($attachements as $attachement) {
                    if (file_exists(public_path($attachement->path))) {
                        @unlink(public_path($attachement->path));
                    }
                    $attachement->delete();
                }
            }
            //save the rating / review
            $review =   ProductReview::updateOrCreate(
                [
                    'product_id' => $productId,
                    'customer_id' => $customerId,
                    'order_id' => $orderId
                ],
                [
                    'rating' => $request->rating,
                    'review' => $request->review,
                    'verified_customer' => 1,
                    'status' => 0,
                ],
            );
            //handle attachements
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $imagePath =  CommonHelper::fileUpload($file, REVIEW_FILE_PATH);
                    $getExtension = strtolower($file->getClientOriginalExtension());
                    //determine the type of file
                    if (in_array($getExtension, CommonHelper::allowedImageExtension())) {
                        $type = 'image';
                    } elseif (in_array($getExtension, CommonHelper::allowedVideoExtension())) {
                        $type = 'video';
                    } else {
                        continue; //skip invalid file
                    }
                    $review->attachments()->create(['path' => $imagePath, 'type' => $type]);
                }
            }
            //check created or updated differentiate
            $isNew = !$existingReview;
            //save notification 
            $notifyRoles = CommonHelper::$generalNotifyRoles;
            $admins = User::role($notifyRoles)->get();
            $title = $isNew ? __('review_submitted_n') : __('review_updated_n');
            $message = $isNew ? __('review_submitted_message_n') : __('review_updated_message_n');
            foreach ($admins as $admin) {
                NotificationHelper::saveNotification(
                    $customerId,
                    CUSTOMER,
                    $admin->id,
                    ADMIN,
                    $isNew ? REVIEW_SUBMITTED : REVIEW_UPDATED,
                    $title,
                    $message,
                    ['review_id' => $review->id, 'url' => route('admin.product.review.show', $review->id)]
                );
            }

            //fire event
            event(new \App\Events\AdminNotificationEvent([
                'message' => $message,
                'title'   => $title,
                'url' => route('admin.product.review.show', $review->id)
            ]));
            DB::commit();
            $message = $isNew ? __('review_submitted') : __('review_updated');
            return CommonHelper::apiResponse(200, $message);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //show existing review and rating
    public function show(Request $request)
    {
        $customerId = auth()->id();
        try {
            $validation = Validator::make($request->all(), [
                'product_id' => 'required|integer|exists:products,id',
                'order_id' => 'required|integer|exists:orders,id'
            ]);
            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors());
            }
            $review = ProductReview::with(['attachments'])
                ->where('product_id', $request->product_id)
                ->where('customer_id', $customerId)
                ->where('order_id', $request->order_id)
                ->first();
            if (!$review) {
                return CommonHelper::apiResponse(400, __('review_not_found'));
            }
            $data = [
                'rating' => number_format($review->rating, 1),
                'review' => $review->review,
                'message' => null,
                'attachments' => [],
                'status' => CommonHelper::$approvalStatuses[$review->status]
            ];
            if ($review->status == 0) {
                $data['message'] =  __('review_pending');
            } elseif ($review->status == 2) {
                $data['message'] =  __('review_rejected');
            }
            //attachements send if it is exist
            $data['attachments'] = $review->attachments ? $review->attachments->map(function ($media) {
                return [
                    'id' => $media->id,
                    'attachement' => asset($media->path)
                ];
            }) : [];



            return CommonHelper::apiResponse(200, __('review_retrieved'),  $data);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
}
