<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\ProductService;
use Illuminate\Http\Request;
use Throwable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Helpers\CommonHelper;
use App\Models\Product;
use App\Models\ProductReview;

class ProductController extends Controller
{
    //view all products with type request
    public function viewAllProducts(Request $request, ProductService $service)
    {
        $validation = Validator::make($request->all(), [
            'type' => 'nullable|in:new,featured,top_selling',
            'keyword' => 'nullable|string|max:100',
            'vehicle_type_id' => 'nullable|exists:vehicle_types,id,status,1',
            'vehicle_make_id' => 'nullable|exists:vehicle_makes,id,status,1',
            'vehicle_model_id' => 'nullable|exists:vehicle_models,id,status,1',
            'sort_by' => 'nullable|in:price_low_to_high,price_high_to_low,rating_low_to_high,rating_high_to_low,latest',
            'current_page' => 'required|integer|min:1',
        ]);

        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors()); // Bad Request
        }
        try {
            $filters = $request->only(['keyword', 'vehicle_type_id', 'vehicle_make_id', 'vehicle_model_id', 'sort_by']);
            $type = $request->type;
            $currencyCode = auth()->user()->currency;
            //pagination
            $perPage = 10;
            $currentPage = $request->current_page;
            $skip = ($currentPage - 1) * $perPage;
            $pagination = ['skip' => $skip, 'take' => $perPage];



            // Apply type
            if (!empty($type)) {
                switch ($type) {
                    case 'new':

                        $products = $service->getLatestProductsForCustomer($currencyCode, null, $pagination, $filters);
                        break;
                    case 'featured':
                        $products = $service->getFeaturedProductsForCustomer($currencyCode, null, $pagination, $filters);
                        break;
                    case 'top_selling':
                        $products = $service->getTopSellingProductsForCustomer($currencyCode, null, $pagination, $filters);
                        break;
                }
            } else {
                // no type passed → get all products with search + filters
                $products = $service->getAllProductsForCustomer($currencyCode, null, $pagination, $filters);
            }


            if (empty($products)) {
                return CommonHelper::apiResponse(400, __('no product found'));
            }

            return CommonHelper::apiResponse(200, 'Products retrieved successfully', $products);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function productDetail(Request $request, ProductService $service)
    {
        $validation = validator::make($request->all(), [
            'id' => 'required|integer|exists:products,id',
        ]);

        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors()); // Bad Request
        }
        try {
            $currencyCode = auth()->user()->currency;
            $product = Product::with('images')->find($request->id);


            if (!$product) {
                return CommonHelper::apiResponse(400, 'Product not found');
            }
            $productWithCurrency = $service->getCurrency($product, $currencyCode);
            $images = [];
            if ($productWithCurrency['featured_image']) {
                $images[] = $productWithCurrency['featured_image'];
            }
            if ($product->images) {
                foreach ($product->images as $image) {
                    $images[] = asset($image->image);
                }
            }
            //get review
            $review = ProductReview::with(['attachments', 'customer:id,full_name,image'])
                ->where('product_id', $product->id)
                ->where('status', 1)
                ->orderBy('rating', 'desc')
                ->take(5)
                ->get();


            $reviewData = $review->map(function ($review) {
                return   [
                    'rating' => $review->rating,
                    'review' => $review->review,
                    'customer_name' => $review->customer?->full_name,
                    'customer_image' => $review->customer?->image ? asset($review->customer->image) : asset(DEFAULT_PATH),
                    'review_media' => $review->attachments?->map(function ($media) {
                        return  asset($media->path);
                    }),
                ];
            });
            $data = [
                'product_id' => $productWithCurrency['id'],
                'product_name' => $productWithCurrency['name'],
                'description' => $productWithCurrency['description'],
                'price' => round($productWithCurrency['price'], 2),
                'stock_in_hand' => $productWithCurrency['stock_in_hand'],
                'out_of_stock' => $productWithCurrency['out_of_stock'],
                'formatted_price' => number_format($productWithCurrency['price'], 2),
                'currency_symbol' => $productWithCurrency['currency_symbol'],
                'images' => $images,
                'rating' => $productWithCurrency['rating'],
                'review' => $reviewData,
            ];
            return CommonHelper::apiResponse(200, 'Product details retrieved successfully', $data);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //display all reviews of product
    public function productReviews(Request $request)
    {
        $validation = validator::make($request->all(), [
            'product_id' => 'required|integer|exists:products,id',
            'current_page' => 'required|min:1|integer'
        ]);

        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors()); // Bad Request
        }
        try {
            //pagination
            $perPage = 10;
            $currentPage = $request->current_page;
            $skip = ($currentPage - 1) * $perPage;
            $product = Product::with('featuredImage:product_id,image')
                ->select('id', 'name', 'rating')
                ->find($request->product_id);


            if (!$product) {
                return CommonHelper::apiResponse(400, 'Product not found');
            }


            $reviewData = ProductReview::with(['attachments', 'customer:id,full_name,image'])
                ->where('product_id', $request->product_id)
                ->skip($skip)
                ->take($perPage)
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($review) {
                    return  [
                        'rating' => $review->rating,
                        'review' => $review->review,
                        'customer_name' => $review->customer?->full_name,
                        'customer_id' => $review->customer?->id,
                        'customer_image' => $review->customer?->image ? asset($review->customer->image) : asset(DEFAULT_PATH),
                        'review_media' => $review->attachments?->map(function ($media) {
                            return  asset($media->path);
                        }),
                    ];
                });

            $data = [
                'product_id' => $product->id,
                'product_name' => $product->name,
                'featured_image' =>  $product->featuredImage?->image ? asset($product->featuredImage->image) : null,
                'rating' => $product->rating,
                'review' => $reviewData,
            ];
            return CommonHelper::apiResponse(200, 'Product Reviews retrieved successfully', $data);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
}
