<?php

namespace App\Http\Controllers\Api;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\Banner;
use App\Models\Notification;
use App\Models\Product;
use App\Models\VehicleMake;
use App\Models\VehicleModel;
use App\Models\VehicleType;
use App\Services\ProductService;
use Illuminate\Http\Request;
use Throwable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Validator as ValidationValidator;

class HomeController extends Controller
{
    protected $productService;
    public function __construct(ProductService $productService)
    {
        $this->productService = $productService;
    }
    public function dashboard(ProductService $service, Request $request)
    {

        $currencyCode = auth()->user()->currency;
        //dd($currencyCode);
        DB::beginTransaction();
        try {
            $banners = Banner::where('status', 1)->orderBy('sort_order', 'asc')->limit(3)->get()->map(function ($banner) {
                $banner->media_path = asset($banner->media_path);
                unset($banner->video_poster);
                return $banner;
            });
            DB::commit();
            $data = [
                'banners' => $banners,
                'new_arrivals' => $service->getLatestProductsForCustomer($currencyCode, 2),
                'feaured_products' => $service->getFeaturedProductsForCustomer($currencyCode, 2),
                'top_selling_products' => $service->getTopSellingProductsForCustomer($currencyCode, 2),
            ];
            return CommonHelper::apiResponse(200, 'Dashboard data retrieved successfully', $data);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //get vehicle types
    public function vehicleTypes()
    {
        try {
            //get data
            $vehicleTypes = VehicleType::where('status', 1)
                ->orderBy('name', 'asc')->get();
            if (!$vehicleTypes) {
                return CommonHelper::apiResponse(400, __('no vehcile type found'));
            }
            return CommonHelper::apiResponse(200, __('vehicle type data retrieved'), $vehicleTypes);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //get vehicle makes
    public function vehicleMake(Request $request)
    {
        $validation = Validator::make($request->all(), [
            'vehicle_type_id' => 'required|exists:vehicle_types,id',
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        try {
            //get data
            $vehicleMakes = VehicleMake::whereHas('types', function ($q) use ($request) {
                $q->where('vehicle_type_id', $request->vehicle_type_id);
            })
                ->with(['types' => function ($q) use ($request) {
                    $q->where('vehicle_type_id', $request->vehicle_type_id)
                        ->select('vehicle_types.id');
                }])
                ->where('status', 1)
                ->select('id', 'name')
                ->orderBy('name', 'asc')
                ->get();
            if ($vehicleMakes->isEmpty()) {
                return CommonHelper::apiResponse(400, __('no vehcile make found'));
            }

            $data = $vehicleMakes->map(function ($make) {
                return [
                    'id' => $make->id,
                    'name' => $make->name,
                    'vehicle_type_id' => $make->types->first()->id ?? null,
                ];
            });
            return CommonHelper::apiResponse(200, __('vehicle make data retrieved'),  $data);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //get vehicle models
    public function vehicleModel(Request $request)
    {
        $validation = Validator::make($request->all(), [
            'vehicle_make_id' => 'required|exists:vehicle_makes,id',
            'vehicle_type_id' => 'required|exists:vehicle_types,id',
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        try {
            //get data
            $vehicleModels = VehicleModel::where('vehicle_make_id', $request->vehicle_make_id)
                ->where('vehicle_type_id', $request->vehicle_type_id)
                ->where('status', 1)
                ->select('id', 'name', 'vehicle_type_id', 'vehicle_make_id')
                ->orderBy('name', 'asc')
                ->get();
            if ($vehicleModels->isEmpty()) {
                return CommonHelper::apiResponse(400, __('no vehcile model found'));
            }
            return CommonHelper::apiResponse(200, __('vehicle model data retrieved'), $vehicleModels);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    //filter product api
    public function filteredProducts(Request $request)
    {
        $validation = Validator::make($request->all(), [
            'keyword' => 'nullable|string|max:100',
            'vehicle_type_id' => 'nullable|exists:vehicle_types,id,status,1',
            'vehicle_make_id' => 'nullable|exists:vehicle_makes,id,status,1',
            'vehicle_model_id' => 'nullable|exists:vehicle_models,id,status,1',
            'sort_by' => 'nullable|in:price_low_to_high,price_high_to_low,rating_low_to_high,rating_high_to_low,latest',
            'current_page' => 'required|integer|min:1',
        ]);
        if ($validation->fails()) {
            return CommonHelper::apiResponse(400, $validation->errors());
        }
        try {
            $query = Product::query()->with(['compatibleModels.vehicleMake', 'compatibleModels.vehicleType', 'featuredImage']);
            //search by keywrod
            if ($request->filled('keyword')) {
                $keyword = $request->keyword;
                $query->where('status', 1)
                    ->where(function ($q) use ($keyword) {
                        $q->where('name', 'like', "{$keyword}%");
                    });
            }
            // filter by vehicle type
            if ($request->filled('vehicle_type_id')) {
                $query->whereHas('compatibleModels.vehicleType', function ($q) use ($request) {
                    $q->where('id', $request->vehicle_type_id)
                        ->where('status', 1);
                });
            }
            //filter by vehicle make
            if ($request->filled('vehicle_make_id')) {
                $query->whereHas('compatibleModels.vehicleMake', function ($q) use ($request) {
                    $q->where('id', $request->vehicle_make_id)
                        ->where('status', 1);
                });
            }
            //filter by vehicle model
            if ($request->filled('vehicle_model_id')) {
                $query->whereHas('compatibleModels', function ($q) use ($request) {
                    $q->where('id', $request->vehicle_model_id)
                        ->where('status', 1);
                });
            }
            //sort by
            if ($request->filled('sort_by')) {
                switch ($request->sort_by) {
                    case 'price_low_to_high':
                        $query->orderBy('price_in_usd', 'asc');
                        break;
                    case 'price_high_to_low':
                        $query->orderBy('price_in_usd', 'desc');
                        break;
                    case 'rating_low_to_high':
                        $query->orderBy('rating', 'asc');
                        break;
                    case 'rating_high_to_low':
                        $query->orderBy();
                        break;
                    default:
                        $query->latest();
                }
            } else {
                $query->latest();
            }

            //pagination
            $perPage = 10;
            $currentPage = $request->current_page;
            $skip = ($currentPage - 1) * $perPage;
            $products = $query->skip($skip)->take($perPage)->get();


            //get currency
            $currencyCode = auth()->user()->currency ?? USD;
            $data = $products->map(function ($product) use ($currencyCode) {
                return $this->productService->getCurrency($product, $currencyCode);
            });

            return CommonHelper::apiResponse(200, __("products retrieved successfully"), $data);
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
    public function seedCurrencyMasetr()
    {
        $countries = ['CN', 'MM', 'US'];
        $data = DB::table('countries')
            ->whereIn('iso2', $countries)
            ->select('iso2 as country_code', 'name as country_name', 'currency as currency_code', 'currency_name as currency_name', 'currency_symbol')
            ->get()
            ->map(function ($row) {
                return (array)$row;
            })->toArray();
        DB::table('currencies_master')->truncate();
        DB::table('currencies_master')->insert($data);
    }
    //get notifications
    public function getNotifications(Request $request)
    {
        $perPage = 10;
        $currentPage = $request->current_page;
        $skip = ($currentPage - 1) * $perPage;
        $notifications = Notification::where('receiver_id', auth()->id())
            ->orderBy('created_at', 'desc')
            ->skip($skip)->take($perPage)
            ->get()
            ->map(function ($notification) {
                $data = json_decode($notification->data, true) ?? [];

                // Merge data keys into notification object
                foreach ($data as $key => $value) {
                    $notification->$key = $value;
                }

                // Optionally remove original data column if you don't want it
                unset($notification->data);

                return $notification;
            });
        if ($notifications->isEmpty()) {
            return CommonHelper::apiResponse(200, __("notifications_empty"), $notifications, '', true);
        }
        return CommonHelper::apiResponse(200, __("notifications_retrieved"), $notifications);
    }
    //mark notification read
    public function markNotificationRead(Request $request)
    {
        try {
            $validation = Validator::make($request->all(), [
                'notification_id' => 'required|exists:notifications,id'
            ]);
            if ($validation->fails()) {
                return CommonHelper::apiResponse(400, $validation->errors());
            }
            $notification = Notification::where('id', $request->notification_id)
                ->where('receiver_id', auth()->id())
                ->first();
            if (!$notification) {
                return CommonHelper::apiResponse(200, __("notifications_not_found"));
            }
            $notification->is_read = 1;
            $notification->save();
            return CommonHelper::apiResponse(200, __("notifications_mark_read"));
        } catch (Throwable $e) {
            return CommonHelper::apiErrorResponse($e);
        }
    }
}
