<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Throwable;
use App\Helpers\CommonHelper;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class ProfileController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:profile-setting')->only(['editProfile', 'updateProfile']);
    }
    public function editProfile()
    {
        $user = Auth::user();
        return view('admin.profile.profile', compact('user'));
    }
    public function updateProfile(Request $request)
    {
        DB::beginTransaction();
        try {
            $user = Auth::user();

            // Base validation rules
            $rules = [
                'name' => 'required|string|max:255',
                'gender' => 'nullable|in:male,female,other',
                'dob' => 'nullable|date',
                'address' => 'nullable|string',
                'status' => 'required|in:0,1',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'password' => 'nullable|min:6|confirmed',

            ];

            if ($request->email !== $user->email) {
                $rules['email'] = 'required|email|unique:users,email';
            }

            if ($request->mobile_number !== $user->mobile_number) {
                $rules['mobile_number'] = 'required|digits_between:8,15|unique:users,mobile_number';
            }
            $validation = Validator::make($request->all(), $rules);
            if ($validation->fails()) {
                return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
            }
            $validatedData = $validation->validated($rules);
            if ($request->filled('dob') && $request->dob !== $user->dob) {
                $dob = Carbon::createFromFormat('d-m-Y', $validatedData['dob'])->format('Y-m-d');
            } else {
                $dob = $user->dob;
            }
            $user->name = $validatedData['name'];
            $user->gender = $validatedData['gender'];
            $user->dob = $dob;
            $user->address = $validatedData['address'];
            $user->status = $validatedData['status'];

            if (isset($validatedData['email'])) {
                $user->email = $validatedData['email'];
            }

            if (isset($validatedData['mobile_number'])) {
                $user->mobile_number = $validatedData['mobile_number'];
            }

            if (!empty($validatedData['password'])) {
                $user->password = Hash::make($validatedData['password']);
            }

            if ($request->hasFile('image')) {
                $image = CommonHelper::fileUpload($request->image, EMPLOYEE_IMAGE_PATH,  $user->image);
                // delete old image if exists
                $oldImagePath = public_path(EMPLOYEE_IMAGE_PATH . '/' . $user->image);
                if ($user->image && file_exists($oldImagePath)) {
                    unlink($oldImagePath);
                }


                $user->image = $image;
            }

            $user->save();

            DB::commit();
            $extra = [
                'redirect' => route('admin.profile')
            ];

            return CommonHelper::jsonResponseWeb(200, 'Profile updated successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();

            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
}
