<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\CommonHelper;
use App\Helpers\NotificationHelper;
use App\Helpers\OrderNotificationHelper;
use App\Http\Controllers\Controller;
use App\Models\Notification;
use App\Models\order;
use App\Models\OrderCancellation;
use App\Models\OrderReturn;
use App\Models\OrderStatusLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Throwable;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\QueryException;
use Illuminate\support\Facades\Auth;

class OrderController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list-order')->only('index');
        $this->middleware('permission:view-order-detail-order|status-update-order')->only(['show']);
        $this->middleware('permission:status-update-order')->only(['refundOrder', 'updateOrderStatus']);
    }
    public function index(Request $request)
    {
        $orders = (new order())->getPaginatedData($request);

        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.order.pagination', compact('orders'))->render(),
            ]);
        }
        //set readed notification to 0
        Notification::where('receiver_type', ADMIN)
            ->where('receiver_id', auth()->id())
            ->whereIn('type', [ORDER_PLACED, ORDER_CANCELLED, ORDER_RETURNED])
            ->where('is_read', 0)
            ->update(['is_read' => 1]);
        return view('admin.order.index', compact('orders'));
    }
    public function show($id)
    {
        $order = Order::with(['customer', 'orderDetails', 'orderStatusLogs', 'transaction', 'shippingAddress', 'billingAddress', 'orderCancel', 'orderReturn'])->findOrFail($id);
        if (!CommonHelper::canViewOrderDetail(Auth::user(), $order)) {
            abort(403, 'You do not have access to view this order.');
        }
        $orderCount = order::where('customer_id', $order->customer_id)->count();
        return view('admin.order.show', compact('order', 'orderCount'));
        //return view('admin.order.show2');
    }

    public function refundOrder(Request $request, $id)
    {
        DB::beginTransaction();


        try {
            $order = order::findOrFail($id);
            if ($request->mark_as_refunded != 2) {
                return CommonHelper::jsonResponseWeb(400, 'Invalid refund request.');
            }
            switch ($order->order_status) {
                case 4: //cancel
                    OrderCancellation::where('order_id', $id)
                        ->update([
                            'refund_status' => $request->mark_as_refunded,
                            'refund_at'    => now(),
                        ]);
                    OrderNotificationHelper::sendRefundNotification($order, ORDER_CANCEL);
                    break;
                case 5: //return
                    OrderReturn::where('order_id', $id)
                        ->update([
                            'refund_status' => $request->mark_as_refunded,
                            'refund_at'    => now(),
                        ]);
                    OrderNotificationHelper::sendRefundNotification($order, ORDER_RETURN);
                    break;
                case 7: //reject
                    $order->refund_status = 2;
                    $order->updated_at   = now();
                    $order->save();
                    OrderNotificationHelper::sendRefundNotification($order, ORDER_REJECT);
                    break;
                default:
                    return CommonHelper::jsonResponseWeb(
                        400,
                        'Refund can only be applied to Cancelled, Returned, or Rejected orders.'
                    );
            }
            // Log refund as status 6 for timeline
            OrderStatusLog::create([
                'order_id' => $id,
                'status' => 6, // 6 = Refund
                'status_date_time' => now(),
            ]);

            $message = "Order refunded successfully.";

            DB::commit();
            $extra = [
                'redirect' => route('admin.orders.show', $id),
            ];
            return CommonHelper::jsonResponseWeb(200, $message, [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function updateOrderStatus(Request $request, $id)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'order_status' => 'integer|required|in:0,1,2,3,4,7',
            'refund_status' => 'integer|nullable|in:1,2',

        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        try {
            $order = order::findOrFail($id);
            $newStatus = (int) $request->order_status;
            $currentOrderStatus = $order->order_status;
            $currentPaymentStatus = $order->payment_status;

            //validation checks
            //check payment is done 
            /*if ($currentPaymentStatus !== 1) {
                return CommonHelper::jsonResponseWeb(
                    400,
                    'Order status can only be updated when payment is completed.'
                );
            }*/
            if ($newStatus === 7 && $currentOrderStatus !== 0) { // Reject

                return CommonHelper::jsonResponseWeb(
                    400,
                    'Order can only be rejected when status is Pending.'
                );
            }


            // Cancel
            if ($newStatus === 4) {
                if (!in_array($currentOrderStatus, [1, 2])) {
                    return CommonHelper::jsonResponseWeb(
                        400,
                        'Order can only be cancelled when status is  Confirmed  or Shipped.'
                    );
                }

                // Insert cancellation record immediately
                OrderCancellation::firstOrCreate(
                    ['order_id' => $order->id],
                    [
                        'order_id'       => $id,
                        'cancel_by_id'   => auth()->id(),
                        'cancel_by_type' => 'admin',
                        'reason'         => $request->cancel_reason ?? null,
                        'refund_status'      => 1,
                        'status'         => 1,
                        'approved_by_id' => auth()->id(),
                        'approved_at' => now()

                    ]
                );
            }
            //shift refund dttaus from not refund to pending (1) if requested order status is rejected
            if ($request->refund_status) {
                $order->refund_status = $request->refund_status;
            }
            $order->order_status = (int) $newStatus;
            $order->save();

            // Log regular status update
            OrderStatusLog::create([
                'order_id' => $id,
                'status' => (int) $request->order_status,
                'status_date_time' => now(),
            ]);

            //save the notification
            OrderNotificationHelper::sendOrderStatusNotification(
                $order,
                (int) $newStatus,
                ADMIN
            );

            DB::commit();
            $message = "Order status updated successfully.";
            $extra = [
                'redirect' => route('admin.orders.show', $id),
            ];
            return CommonHelper::jsonResponseWeb(200, $message, [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    //update returned orders
    public function updateReturnOrder(Request $request, $id)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'status' => 'integer|required|in:1,2',
            'refund_status' => 'integer|nullable|in:1,2',

        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        $returnedOrder = OrderReturn::findOrFail($id);
        //get related order
        $order = order::findOrFail($returnedOrder->order_id);
        try {
            $returnedOrder->status = $request->status;
            //check if status is approved then switch its refund status to pending getting from request
            if ($request->status) {
                $returnedOrder->refund_status = $request->refund_status ?? 0;
            }
            $returnedOrder->approved_by_id =  auth()->id();
            $returnedOrder->approved_at    = now();
            $returnedOrder->save();
            $message = "Return Order updated successfully.";
            OrderNotificationHelper::sendCancelReturnStatusesNotification($order, $request->status, 'return');
            DB::commit();

            $extra = [
                'redirect' => route('admin.orders.show', $returnedOrder->order_id),
            ];
            return CommonHelper::jsonResponseWeb(200, $message, [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    //update cancelled orders
    public function updateCancelOrder(Request $request, $id)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'status' => 'integer|required|in:1,2',
            'refund_status' => 'integer|nullable|in:1,2',

        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        $cancelledOrder = OrderCancellation::findOrFail($id);
        //get related order
        $order = order::findOrFail($cancelledOrder->order_id);
        try {
            $cancelledOrder->status = $request->status;
            //check if status is approved then switch its refund status to pending getting from request
            if ($request->status) {
                $cancelledOrder->refund_status = $request->refund_status ?? 0;
            }
            $cancelledOrder->approved_by_id =  auth()->id();
            $cancelledOrder->approved_at    = now();
            $cancelledOrder->save();
            $message = "Cancel Order updated successfully.";
            OrderNotificationHelper::sendCancelReturnStatusesNotification($order, $request->status, 'cancel');
            DB::commit();

            $extra = [
                'redirect' => route('admin.orders.show', $cancelledOrder->order_id),
            ];
            return CommonHelper::jsonResponseWeb(200, $message, [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    //mark items received
    public function itemsReceived(Request $request, $id)
    {
        $validation = Validator::make($request->all(), [
            'items_received' => 'required|in:0,1',
        ]);

        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }

        try {
            $order = Order::findOrFail($id);

            if (!in_array($order->order_status, [4, 5])) {
                return CommonHelper::jsonResponseWeb(400, 'Items can only be marked as received for Cancelled or Returned orders.');
            }

            if ($order->order_status == 4) { // Cancel
                $cancelledOrder = OrderCancellation::where('order_id', $id)->first();
                if (!$cancelledOrder) {
                    return CommonHelper::jsonResponseWeb(404, 'Cancellation record not found.');
                }

                $cancelledOrder->items_received = $request->items_received;
                $cancelledOrder->items_received_at = now();
                $cancelledOrder->save();
            }

            if ($order->order_status == 5) { // Return
                $return = OrderReturn::where('order_id', $id)->first();
                if (!$return) {
                    return CommonHelper::jsonResponseWeb(404, 'Return record not found.');
                }

                $return->goods_received = $request->items_received;
                $return->goods_received_at = now();
                $return->save();
            }

            $extra = [
                'redirect' => route('admin.orders.show', $id),
            ];
            $message = __('items_marked_received');
            return CommonHelper::jsonResponseWeb(200, $message, [], $extra);
        } catch (Throwable $e) {
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
}
