<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class VehicleMake extends Model
{
    use HasFactory;
    use SoftDeletes;
    protected $fillable = [
        'vehicle_type_id',
        'name',
        'image_path',
        'status'
    ];
    public function models()
    {
        return $this->hasMany(VehicleModel::class);
    }
    public function types()
    {
        return $this->belongsToMany(VehicleType::class, 'vehicle_make_vehicle_type')
            ->withPivot('vehicle_type_id');
    }
    public function getPaginatedData($request)
    {
        $query = self::withTrashed()->whereIn('status', [0, 1]);

        $perPage = $request->input('per_page', 10);
        $sortBy = $request->get('sort_by');
        $allowedSortColumns = ['id', 'name', 'status', 'created_at']; // add only valid DB columns here

        // Default to 'vehicle_makes.id' if invalid or empty
        $sortBy = in_array($sortBy, $allowedSortColumns) ? 'vehicle_makes.' . $sortBy : 'vehicle_makes.id';

        $sortOrder = $request->get('sort_order', 'desc');
        $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? $sortOrder : 'desc';

        if ($request->has('keyword') && $request->get('keyword') != '') {
            $keyword = $request->keyword;
            $query->where(function ($q) use ($keyword) {
                $q->where('name', 'like', "%$keyword%")
                    ->orWhere('status', 'like', "%$keyword%")
                    ->orWhere('id', 'like', "%$keyword%");
            });
        }

        if ($request->has('status') && $request->get('status') != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('vehicle_type') && $request->get('vehicle_type') != '') {
            $query->whereHas('types', function ($q) use ($request) {
                $q->where('vehicle_types.id', $request->get('vehicle_type'));
            });
        }

        return $query->orderBy($sortBy, $sortOrder)->paginate($perPage);
    }
    protected static function booted()
    {
        static::deleting(function ($vehicleMake) {
            // Detach from pivot
            $vehicleMake->types()->detach();

            // Delete related vehicle models
            $vehicleMake->models()->each(function ($model) {
                $model->status = 2;
                $model->save();
                $model->delete();
            });
        });
    }
}
