<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\VehicleMake;
use App\Models\VehicleModel;
use App\Models\VehicleType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Throwable;
use Illuminate\Database\QueryException;

class VehicleModelController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list-vehicle_model')->only('index');
        $this->middleware('permission:add-vehicle_model')->only(['create', 'store', 'getVehicleTypes']);
        $this->middleware('permission:edit-vehicle_model')->only(['edit', 'update', 'getVehicleTypes']);
        $this->middleware('permission:edit-vehicle_model')->only(['edit', 'show', 'getVehicleTypes']);
        $this->middleware('permission:delete-vehicle_model')->only(['destroy']);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $vehicleModels = (new VehicleModel())->getPaginatedData($request);
        $vehicleMakes = VehicleMake::where('status', 1)->get();
        //dd($vehicleModels);
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.vehicleModel.pagination', compact('vehicleModels', 'vehicleMakes'))->render(),
            ]);
        }
        return view('admin.vehicleModel.index', compact('vehicleModels', 'vehicleMakes'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $vehicleMakes = VehicleMake::where('status', 1)->get();
        $vehicleTypes = VehicleType::where('status', 1)->get();
        return view('admin.vehicleModel.create', compact('vehicleMakes', 'vehicleTypes'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'required|min:3|max:200|string',
            'vehicle_make' => 'required|exists:vehicle_makes,id',
            'vehicle_type' => 'required|exists:vehicle_types,id',
            'status' => 'required|in:0,1'
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        //dd($request->all());
        try {
            VehicleModel::create([
                'vehicle_make_id' => $request->vehicle_make,
                'vehicle_type_id' => $request->vehicle_type,
                'name' => $request->name,
                'status' => $request->status,

            ]);

            DB::commit();
            $extra = [
                'redirect' => route('admin.vehicle-model.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle model added successfully', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $vehicleMakes = VehicleMake::where('status', 1)
            ->orWhereHas('models', function ($q) use ($id) {
                $q->where('id', $id);
            })->get();
        $vehicleModelData = VehicleModel::with('vehicleMake', 'vehicleType')->findOrFail($id);
        $vehicleTypes = $vehicleModelData->vehicleMake ? $vehicleModelData->vehicleMake->types()->where('status', 1)->get() : collect();
        return view('admin.vehicleModel.edit', compact('vehicleMakes', 'vehicleModelData', 'vehicleTypes'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'required|min:3|max:200|string',
            'status' => 'required|in:0,1',
            'vehicle_make' => 'required|exists:vehicle_makes,id',
            'vehicle_type' => 'required|exists:vehicle_types,id',

        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        try {
            $vehicleModel = VehicleModel::findOrFail($id);
            $vehicleModel->update([
                'name' => $request->name,
                'vehicle_make_id' => $request->vehicle_make,
                'vehicle_type_id' => $request->vehicle_type,
                'status' => $request->status
            ]);
            DB::commit();
            $extra = [
                'redirect' => route('admin.vehicle-model.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle Model updated successfully.', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $vehicleModel = VehicleModel::findOrFail($id);
            $vehicleModel->update(['status' => 2]);
            $vehicleModel->delete();

            DB::commit();
            $extra = [
                'redirect' => route('admin.vehicle-model.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'Vehicle Model deleted successfully.', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function getVehicleTypes($makeId)
    {
        $make =  VehicleMake::with(['types' => function ($query) {
            $query->where('status', 1);
        }])->find($makeId);
        if (!$make) {
            return response()->json([
                'status' => 'false',
                'message' => 'Make not found'
            ], 400);
        }
        $types = $make->types->map(function ($type) {
            return [
                'id' => $type->id,
                'name' => $type->name
            ];
        });

        return response()->json([
            'status' => true,
            'data' => $types
        ]);
    }
}
