<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\CommonHelper;
use App\Http\Controllers\Controller;
use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Throwable;
use Illuminate\Database\QueryException;

class DepartmentController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list-department')->only('index');
        $this->middleware('permission:add-department')->only(['create', 'store', 'checkDuplicateDepartment']);
        $this->middleware('permission:edit-department')->only(['edit', 'update', 'checkDuplicateDepartment']);
        $this->middleware('permission:edit-department')->only(['edit', 'show']);
        $this->middleware('permission:delete-department')->only(['destroy']);
    }

    private function checkDuplicateDepartment($name, $ignoreId = null)
    {
        $normalizedInput = CommonHelper::normalizeDepartmentName($name);

        // Fetch all existing departments
        $departments = Department::all();

        foreach ($departments as $department) {
            $normalizedExisting = CommonHelper::normalizeDepartmentName($department->name);

            // If normalized names match and we're not editing the same ID
            if ($normalizedInput === $normalizedExisting && (!$ignoreId || $ignoreId != $department->id)) {
                $extra = [
                    'redirect' => route('admin.department.index'),
                ];
                return CommonHelper::jsonResponseWeb(400, 'Department already exists!', [], $extra);
            }
        }
    }
    public function index()
    {
        $departments = Department::orderBy('id', 'desc')->get();
        return view('admin.department.index', compact('departments'));
    }
    public function create()
    {
        return view('admin.department.create');
    }
    public function store(Request $request)
    {

        DB::beginTransaction();
        $validation = Validator::make($request->all(), [
            'name' => 'required|string',
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        if ($duplicateResponse = $this->checkDuplicateDepartment($request->name)) {
            return $duplicateResponse;
        }

        try {
            Department::create([
                'name' => $request->name,
            ]);

            DB::commit();
            $extra = [
                'redirect' => route('admin.department.index'),
            ];
            return CommonHelper::jsonResponseWeb(200, 'Department added successfully.', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return  CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function edit($id)
    {
        $department = Department::findOrFail($id);
        if (in_array($department->id, fixedDeptsID())) {
            abort(403, 'This department cannot be edited.');
        }
        return view('admin.department.edit', compact('department'));
    }
    public function update($id, Request $request)
    {

        DB::beginTransaction();
        $department = Department::findOrFail($id);
        $validation = Validator::make($request->all(), [
            'name' => 'required|string',
        ]);
        if ($validation->fails()) {
            return CommonHelper::jsonResponseWeb(400, '', $validation->errors());
        }
        if ($duplicateResponse = $this->checkDuplicateDepartment($request->name, $department->id)) {
            return $duplicateResponse;
        }
        if (in_array($department->id, fixedDeptsID())) {
            abort(403, 'This department cannot be edited.');
        }
        try {
            $department->name =  $request->name;
            $department->save();

            DB::commit();
            $extra = [
                'redirect' => route('admin.department.index'),
            ];
            return CommonHelper::jsonResponseWeb(200, 'Department aupdated successfully.', [], $extra);
        } catch (Throwable $e) {
            DB::rollBack();
            return  CommonHelper::jsonErrorResponseWeb($e);
        }
    }
    public function destroy($id)
    {
        try {
            DB::beginTransaction();
            $department = Department::findOrFail($id);
            if (in_array($department->id, fixedDeptsID())) {
                abort(403, 'This department cannot be deleted.');
            }
            $department->update(['status' => 2]);
            $department->delete();
            DB::commit();
            $extra = [
                'redirect' => route('admin.department.index')
            ];
            return CommonHelper::jsonResponseWeb(200, 'department deleted successfully.', [], $extra);
        } catch (QueryException $e) {
            $msg = $e->getMessage();
            if (strpos($msg, '1451') !== false) { // fk check
                $message = 'Please delete entries first from its relevant sections';
                return response()->json(['status' => false, 'status_code' => 500, 'message' => $message], 500);
            }
        } catch (Throwable $e) {
            DB::rollBack();
            return CommonHelper::jsonErrorResponseWeb($e);
        }
    }
}
